(* ::Subsubsection:: *)
(*BakerSchmidtCameraSM package*)

(* ::Text:: *)
(*This impements the Backer-Schmidt Camera with Spherical Mirrors optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/BakerSchmidtCameraSM"}]*)

BeginPackage["GeometricOptics`Packages`BakerSchmidtCameraSM`", {"GeometricOptics`"}]

Options[BakerSchmidtCameraSM] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
BakerSchmidtCameraSM::usage="BakerSchmidtCameraSM calculates parameters for a Backer-Schmidt Camera with Spherical Mirrors optical system.";

BakerSchmidtCameraSMInputPanel::usage = "BakerSchmidtCameraSMInputPanel gives the GUI panel to work with BakerSchmidtCameraSM";

$BakerSchmidtCameraSMInputVariables = {"f1", "\[Beta]", "diam", "\[Theta]", "sf"};

Begin["`Private`"]

BakerSchmidtCameraSM[{f1_, beta_, diam_, theta_, sf_}, opts___]:= BakerSchmidtCameraSM[f1, beta, diam, theta, sf, opts];

BakerSchmidtCameraSM[f1_, beta_, diam_, theta_, sf_, OptionsPattern[]] := 
 Module[{gamma, delta, a4, a40, rad, c, thick, ind, costasf, as, sol, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["BakerSchmidtCameraSM", {$BakerSchmidtCameraSMInputVariables, {f1, beta, diam, theta, sf}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 2, "PackageName" -> "BakerSchmidtCameraSM", "Arguments" -> <|"f1" -> f1, "\[Beta]" -> beta, "diam" -> diam, "\[Theta]" -> theta, "sf" -> sf|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
			
			gamma = 2;
			(* To eliminate coma *)
			delta0 = N@(-1 - 3 beta + beta^2 + beta^3)/(beta (-2 + beta^2));
			(* To eliminate spherical aberration *)
			a40 = N@(2 beta^2 - beta^4)/(32 f1^3 (-1 + 1.518722));
			(* Final optical system *)
			rad = N@{10^12, 1/c, -2 f1, -gamma f1}; 
			thick = N@{5, delta f1, -beta f1};
			ind = N@{{1, 1.518722, 1, -1, 1}, {1, 1.522829, 1, -1, 1}, {1, 1.514719, 1, -1, 1}};
			costasf = N@{0, {a4, as}, 0, 0};
			
			TotalAberrations[rad, thick, ind, costasf, diam/2, 0, 0, -Infinity, x, theta, {lambda1, lambda2, lambda3}];
			
			sol = FindRoot[{GOaberration[[1]] == sf, GOcoma[[1]] == 0, GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]] == -0.06}, {a4, a40}, {delta, delta0}, {c, -10^-6}, MaxIterations -> 400];

			outputs = {{"Radius of the second surface of corrector", 1/a4 /. sol},
						{"Aspheric constant of corrector", delta /. sol},
						{"Radius of primary mirror", -2 f1},
						{"Radius of secondary mirror", -2 f1},
						{"Distance of primary mirror from corrector", delta0 f1},
						{"Distance between mirrors", -beta f1},
						{"Total focal", GOfocalLength[[1]] /. sol},
						{"Back focal", (GOdistancegauss[[1, GOn]] - beta f1) /. sol},
						{"Height of image", GOimageHeight[[1, GOn]] /. sol},
						{"Spherical aberration", GOaberration[[1]] /. sol},
						{"Coma", GOcoma[[1]] /. sol},
						{"Astigmatism", GOastigmatism[[1]] /. sol},
						{"Curvature", GOcurvature /. sol}}//N;
			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Coefficient \[Beta] = 1-\[Alpha], where \[Alpha] is the obstruction factor (\[Alpha] \[GreaterEqual] 0.40)", "beta", beta},
							{"Diameter of primary mirror", "diam", diam},
							{"Field angle  in degree", "theta", theta},
							{"Residual spherical aberration", "sf", sf}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "BakerSchmidtCameraSM" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "BakerSchmidtCameraSM", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "BakerSchmidtCameraSM" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "BakerSchmidtCameraSM", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
											 {"\[Beta]", beta},
											 {"diam", diam},
											 {"\[Theta]", theta},
											 {"sf", sf}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[
					TextCell[{1/a4 /. sol, c /. sol, -2 f1, -2 f1, delta0 f1, -beta f1, GOfocalLength[[1]] /. sol, (GOdistancegauss[[1, GOn]] -beta f1) /. sol, GOimageHeight[[1, GOn]] /. sol,
							  GOaberration[[1]] /. sol, GOcoma[[1]] /. sol, GOastigmatism[[1]] /. sol, N[GOcurvature /. sol]}, "Output"]]],

  

							
			(* Arguments are not correct *)
			MessageDialog["BakerSchmidtCameraSM not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"]]];

BakerSchmidtCameraSMInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "BakerSchmidtCameraSM"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, beta, diam, theta, sf} = ReplaceAll[$BakerSchmidtCameraSMInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of focal length of the primary mirror"]},
						  {"Coefficient \[Beta] = 1-\[Alpha], where \[Alpha] is the obstruction factor (\[Alpha] \[GreaterEqual] 0.40)", "\[Beta]", Tooltip[InputField[Dynamic[beta], Alignment -> Center], "Insert the value of \[Beta]"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]},
						  {"Residual spherical aberration", "sf", Tooltip[InputField[Dynamic[sf], Alignment -> Center], "Insert the value of residual spherical aberration"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, beta, diam, theta, sf}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, beta, diam, theta, sf}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]